/*:
 * @target MZ
 * @plugindesc v1.7.4 統合版：未習得=Lv→DB順／既習得=上＆右にMP/TP（ワード・色可）／特性スキルの扱いを選択可／1カラム対応
 * @author You
 *
 * @help UpcomingSkillsListMZ.js
 * 既習得（恒久＋※設定で特性スキル）を上、未習得（これから覚える）を下に表示。
 * 右端：未習得は「Lv X」、既習得は「MP/TP」を表示（ワードや色を設定可）。
 * 未習得の並びは (習得Lv, 職業DBのスキル習得リスト行順)。
 *
 * @param UpcomingTextColor
 * @text 未習得スキルの文字色
 * @type string
 * @default #66ccff
 *
 * @param ShowLevelForLearned
 * @text 既習得にもLv表示（MP/TPが無い時）
 * @type boolean
 * @default false
 *
 * @param ForceSingleColumn
 * @text スキルリストを1カラムにする
 * @type boolean
 * @default true
 *
 * @param TraitSkillsHandling
 * @text 特性スキルの扱い
 * @type select
 * @option asLearned（上段・使用可）
 * @value asLearned
 * @option asUpcoming（下段・使用不可）
 * @value asUpcoming
 * @option ignore（無視）
 * @value ignore
 * @default asLearned
 *
 * @param ShowMpForLearned
 * @text 既習得にMP表示
 * @type boolean
 * @default true
 *
 * @param ShowMpWhenZero
 * @text MP0でも表示
 * @type boolean
 * @default false
 *
 * @param MpText
 * @text MPワード
 * @type string
 * @default MP
 *
 * @param MpTextColor
 * @text MP文字色
 * @type string
 * @default
 *
 * @param ShowTpForLearned
 * @text 既習得にTP表示
 * @type boolean
 * @default true
 *
 * @param ShowTpWhenZero
 * @text TP0でも表示
 * @type boolean
 * @default false
 *
 * @param TpText
 * @text TPワード
 * @type string
 * @default TP
 *
 * @param TpTextColor
 * @text TP文字色
 * @type string
 * @default
 */
(() => {
  'use strict';
  const P = PluginManager.parameters('UpcomingSkillsListMZ');

  const UPCOMING_COLOR      = String(P['UpcomingTextColor'] || '#66ccff');
  const SHOW_LV_FOR_LEARNED = P['ShowLevelForLearned'] === 'true';
  const FORCE_ONE_COLUMN    = P['ForceSingleColumn'] === 'true';

  const TRAIT_MODE          = String(P['TraitSkillsHandling'] || 'asLearned'); // asLearned|asUpcoming|ignore

  const SHOW_MP             = P['ShowMpForLearned'] !== 'false';
  const SHOW_MP_ZERO        = P['ShowMpWhenZero'] === 'true';
  const MP_TEXT             = String(P['MpText'] || 'MP');
  const MP_COLOR_PARAM      = String(P['MpTextColor'] || '');

  const SHOW_TP             = P['ShowTpForLearned'] !== 'false';
  const SHOW_TP_ZERO        = P['ShowTpWhenZero'] === 'true';
  const TP_TEXT             = String(P['TpText'] || 'TP');
  const TP_COLOR_PARAM      = String(P['TpTextColor'] || '');

  const FLAG_FIELD  = '_upcomingFlags';   // { skillId: true } 未習得
  const LEVEL_FIELD = '_requiredLevels';  // { skillId: level }

  function allowStype(win, sk){ const st=win._stypeId; return !st || sk.stypeId===st; }
  function resolveColor(v){ if(!v) return null; const n=Number(v); return Number.isNaN(n)? String(v) : ColorManager.textColor(n); }
  const MP_COLOR = resolveColor(MP_COLOR_PARAM);
  const TP_COLOR = resolveColor(TP_COLOR_PARAM);

  function isPermanentlyLearned(actor,id){ return !!(actor && actor._skills && actor._skills.includes(id)); }
  function findLearnLevelInClass(cls,id){
    if (!cls || !cls.learnings) return null;
    const hit = cls.learnings.find(l=>l.skillId===id);
    return hit ? Number(hit.level)||0 : null;
  }

  if (FORCE_ONE_COLUMN) Window_SkillList.prototype.maxCols = function(){ return 1; };

  Window_SkillList.prototype.itemAt = function(i){
    return (this._data && i>=0 && i<this._data.length) ? this._data[i] : null;
  };

  // 中核
  Window_SkillList.prototype.makeItemList = function(){
    const a = this._actor;
    this._data=[]; this[FLAG_FIELD]={}; this[LEVEL_FIELD]={};
    if(!a) return;

    const cls = a.currentClass();
    const learn = (cls && cls.learnings) ? cls.learnings.slice() : [];

    // 既習得（恒久）
    const learned = [];
    for (let i=0;i<a._skills.length;i++){
      const s = $dataSkills[a._skills[i]];
      if (s && allowStype(this,s)) learned.push(s);
    }

    // 特性スキル
    let traitSkills = [];
    if (a.addedSkills) {
      traitSkills = a.addedSkills().map(id=>$dataSkills[id]).filter(s=>s && allowStype(this,s));
    }

    if (TRAIT_MODE === 'asLearned') {
      // 既習得側に混ぜる（重複除去）
      const learnedIds = new Set(learned.map(s=>s.id));
      for (const s of traitSkills) if (!learnedIds.has(s.id)) learned.push(s);
    }

    // 未習得：DBのlearnings基準
    const pairs = [];
    for (let i=0;i<learn.length;i++){
      const L = learn[i];
      const s = $dataSkills[L.skillId];
      if (!s || !allowStype(this,s)) continue;
      if (isPermanentlyLearned(a,s.id)) continue; // 恒久なら上段
      if (TRAIT_MODE === 'asLearned' && traitSkills.some(ts=>ts.id===s.id)) continue; // 特性で習得扱いにしてるなら除外
      pairs.push({s, lv:Number(L.level)||0, idx:i});
    }

    // 特性を未習得として出すモード
    if (TRAIT_MODE === 'asUpcoming') {
      for (let i=0;i<traitSkills.length;i++){
        const s = traitSkills[i];
        // すでに pairs に入っていない＆恒久習得でもない場合だけ追加
        if (isPermanentlyLearned(a,s.id)) continue;
        if (pairs.some(p=>p.s.id===s.id)) continue;
        const lv = findLearnLevelInClass(cls, s.id);
        pairs.push({s, lv:(lv!=null? lv:9999), idx:999999+i});
      }
    }

    // Lv→DB行順で厳密整列
    pairs.sort((a,b)=> a.lv===b.lv ? a.idx-b.idx : a.lv-b.lv);

    // 反映
    const upcoming=[];
    for (const p of pairs){
      this[FLAG_FIELD][p.s.id]=true;
      this[LEVEL_FIELD][p.s.id]=p.lv;
      upcoming.push(p.s);
    }

    this._data = learned.concat(upcoming);
  };

  // 未習得は使用不可（asUpcoming時の特性スキルも使用不可にする）
  Window_SkillList.prototype.isEnabled = function(item){
    const a=this._actor;
    if(!a||!item) return false;
    if (this[FLAG_FIELD][item.id]) return false;
    return a.canUse(item);
  };

  // 右端トークン描画（右寄せ複数）
  function drawRightTokens(win, rect, tokens){
    let x = rect.x + rect.width;
    for (let i=tokens.length-1; i>=0; i--){
      const t = tokens[i]; if (!t || !t.text) continue;
      const w = win.textWidth(t.text);
      x -= w;
      if (t.color) win.changeTextColor(t.color); else win.resetTextColor();
      win.drawText(t.text, x, rect.y, w, 'left');
      x -= 6;
    }
    win.resetTextColor();
  }

  Window_SkillList.prototype.drawItem = function(index){
    const it = this.itemAt(index); if(!it) return;
    const r  = this.itemLineRect(index);
    const up = !!this[FLAG_FIELD][it.id];
    const lv = (this[LEVEL_FIELD][it.id] != null) ? this[LEVEL_FIELD][it.id] : null;

    this.changePaintOpacity(this.isEnabled(it));

    // 右側トークン
    const tokens = [];
    if (up) {
      if (lv!=null && lv!==9999) tokens.push({text:`Lv ${lv}`, color:null});
    } else {
      if (SHOW_MP) {
        const mp = this._actor.skillMpCost(it);
        if (mp>0 || SHOW_MP_ZERO) tokens.push({text:`${MP_TEXT} ${mp}`, color:MP_COLOR});
      }
      if (SHOW_TP) {
        const tp = this._actor.skillTpCost(it);
        if (tp>0 || SHOW_TP_ZERO) tokens.push({text:`${TP_TEXT} ${tp}`, color:TP_COLOR});
      }
      if (!tokens.length && SHOW_LV_FOR_LEARNED) {
        const lv2 = findLearnLevelInClass(this._actor.currentClass(), it.id);
        if (lv2!=null) tokens.push({text:`Lv ${lv2}`, color:null});
      }
    }

    // 名前側カラー
    if (up) this.changeTextColor(UPCOMING_COLOR); else this.resetTextColor();

    // 右側幅計算
    const rightW = tokens.reduce((acc,t)=> acc + this.textWidth(t.text) + 6, 0) - (tokens.length?6:0);
    const iconW  = 36;

    // 本体
    this.drawIcon(it.iconIndex, r.x, r.y+2);
    this.drawText(it.name, r.x+iconW, r.y, r.width - rightW - iconW);

    if (tokens.length) drawRightTokens(this, r, tokens);

    this.resetTextColor();
    this.changePaintOpacity(true);
  };
})();
